#extension GL_ARB_gpu_shader5 : enable

varying vec2 texcoord;

vec2 HashVec(vec2 d)
{
    const vec2 k1 = vec2(123.4, 234.5);
    const vec2 k2 = vec2(234.5, 345.6);
    const float MULT = 58322.51;
    const float TIME_MULT = 2.0;

    vec2 s = sin(vec2(dot(d, k1), dot(d, k2)));
    return sin(s * MULT + frameTimeCounter * TIME_MULT);
}

float Octave(vec2 uv, float gridSize)
{
    // Precompute the scaled coordinate to avoid redundant multiplication.
    vec2 scaledUV = uv * gridSize;
    vec2 gridUV = fract(scaledUV);
    vec2 cellId = floor(scaledUV);
    vec2 smoothUV = smoothstep(0.0, 1.0, gridUV);

    // Precompute wrapped cell IDs for the four corners.
    vec2 cell00 = mod(cellId, gridSize);
    vec2 cell10 = mod(cellId + vec2(1.0, 0.0), gridSize);
    vec2 cell01 = mod(cellId + vec2(0.0, 1.0), gridSize);
    vec2 cell11 = mod(cellId + vec2(1.0, 1.0), gridSize);

    // Precompute the UV offsets for interpolation.
    vec2 uv10 = gridUV - vec2(1.0, 0.0);
    vec2 uv01 = gridUV - vec2(0.0, 1.0);
    vec2 uv11 = gridUV - vec2(1.0, 1.0);

    // Compute dot products with hashed vectors.
    float n00 = dot(HashVec(cell00), gridUV);
    float n10 = dot(HashVec(cell10), uv10);
    float n01 = dot(HashVec(cell01), uv01);
    float n11 = dot(HashVec(cell11), uv11);

    // Perform bilinear interpolation.
    float nx0 = mix(n00, n10, smoothUV.x);
    float nx1 = mix(n01, n11, smoothUV.x);
    return mix(nx0, nx1, smoothUV.y);
}

// Standard fBm-style 2D noise function
float n2d(vec2 p)
{
    vec2 i = floor(p);
    p -= i;
    p *= p * (3. - p * 2.);
    return dot(mat2(fract(sin(mod(vec4(0, 1, 113, 114) + dot(i, vec2(1, 113)), 6.2831853)) * 43758.5453)) * vec2(1. - p.y, p.y), vec2(1. - p.x, p.x));
}

// Combined ocean surface noise function with both fBm and Gabor components
float seaSurf(vec2 pxz)
{
    pxz *= 256.0;

    vec2 mv = frameTimeCounter * vec2(-0.6, -1.5) * 0.5; // Motion vector to animate waves
    float noise = 0.0;

    // Layered fBm-style noise for broader water features
    noise += n2d(pxz * 1.0 - mv) * 0.2;
    noise += n2d(pxz * 0.4 - mv) * 0.2;
    noise += n2d(pxz * 1.6 + mv) * 0.03;
    noise += n2d(pxz * 4.0 - mv) * 0.04;

    return noise * 0.5;
}

float getWaterHeightmap(vec2 posxz)
{
    float scale = 16.0;
    posxz *= scale;

    vec2 pos = posxz;
    vec2 movement = vec2(-0.01 * frameTimeCounter * scale, 0.0);
    float caustic = 0.0;
    float weightSum = 0.0;
    const mat2 rotationMatrix = mat2(vec2(-0.7373937, -0.6754632), vec2(0.6754632, -0.7373937));
    for (int i = 0; i < 4; i++)
    {
        float wave = n2d((pos * vec2(2.0, 1.0) + movement) * exp(i * 1.0));
        caustic += wave * exp(-i * 0.5);
        weightSum += exp(-i * 0.5);
        pos = rotationMatrix * pos;
    }
    return caustic / weightSum;
}

float WaterMap(vec2 pos)
{
    //  return getWaterHeightmap(pos);

    float gradient = 0.0;
    float frequency = 8.0;
    float amplitude = 6.0;

    for (int i = 0; i < 6; i++)
    {
        gradient += (Octave(pos, frequency) + 0.5) / amplitude;
        frequency *= 3.0;
        amplitude *= 1.75;
    }

    return gradient * 6.0;
}
float WaterMapFast(vec2 pos)
{
    // return seaSurf(pos * 512);
    // return getWaterHeightmap(pos);

    float gradient = 0.0;
    float frequency = 8.0;
    float amplitude = 6.0;

    // Use fewer iterations for a faster approximation.
    for (int i = 0; i < 4; i++)
    {
        gradient += (Octave(pos, frequency) + 0.5) / amplitude;
        frequency *= 3.0;
        amplitude *= 1.75;
    }

    return gradient * 6.0;
}

float CausticsMap(float h, vec2 pos)
{
    // Adaptive sampling distance, controlled by rainStrength (0 = dry, 1 = heavy rain)
    float baseOffset = 0.005;
    float maxOffset = 0.10;
    float offset = mix(baseOffset, maxOffset, rainStrength);
    pos *= 0.5;
    // Central difference gradient: more balanced and reduces directional bias
    float hX1 = WaterMapFast(pos + vec2(offset, 0.0));
    float hX2 = WaterMapFast(pos - vec2(offset, 0.0));
    float hY1 = WaterMapFast(pos + vec2(0.0, offset));
    float hY2 = WaterMapFast(pos - vec2(0.0, offset));

    vec2 gradient = vec2(hX1 - hX2, hY1 - hY2) / (2.0 * offset);

    // Gradient length controls caustic sharpness
    float gradLength = length(gradient);

    // Smooth inverse-square falloff, adjusted for rain
    float baseFalloff = 0.05;
    float rainFalloff = 0.5;
    float falloff = mix(baseFalloff, rainFalloff, rainStrength);
    float intensity = 1 / (gradLength * gradLength + falloff);

    // Final caustics brightness: clamped and softened
    float caustic = pow(intensity, 0.25) * 1.0;
    return clamp(caustic, 0.0, 1.0 - rainStrength * 0.9);
}

void main()
{
    /* DRAWBUFFERS:7*/
    float water = WaterMap(texcoord);
    gl_FragData[0].r = water;

#ifdef CAUSTICS
    float casutic = CausticsMap(water, texcoord);
    gl_FragData[0].g = casutic;
#endif
}
