#extension GL_EXT_gpu_shader4 : enable
#extension GL_ARB_gpu_shader5 : enable
#extension GL_ARB_texture_gather : enable


#define TAA_USE_YCOCG

#if MC_GL_VENDOR_INTEL || MC_GL_VENDOR_OTHER
    #define USE_GATHER 0
#else
    #define USE_GATHER 1
#endif

varying vec2 texcoord;

vec3 RGBtoYCoCg(vec3 RGB)
{
    const mat3 RGBToYCoCgMatrix = mat3(0.25, 0.5, -0.25, 0.5, 0.0, 0.5, 0.25, -0.5, -0.25);

    return RGBToYCoCgMatrix * RGB;
}

vec3 YCoCgtoRGB(vec3 YCoCg)
{
    const mat3 YCoCgToRGBMatrix = mat3(1.0, 1.0, 1.0, 1.0, 0.0, -1.0, -1.0, 1.0, -1.0);

    return YCoCgToRGBMatrix * YCoCg;
}
// ---------- colour-space gateway ----------
#if TAA_USE_YCOCG
    #define ENC_COL(c) RGBtoYCoCg(c)
    #define DEC_COL(c) YCoCgtoRGB(c)
#else
    #define ENC_COL(c) (c)
    #define DEC_COL(c) (c)
#endif

// Fetch history pixel

// Cubic Hermite interpolation
vec3 CubicHermite(vec3 A, vec3 B, vec3 C, vec3 D, float t)
{
    float t2 = t * t;
    float t3 = t2 * t;
    return ((-0.5 * A + 1.5 * B - 1.5 * C + 0.5 * D) * t3) + ((A - 2.5 * B + 2.0 * C - 0.5 * D) * t2) + ((-0.5 * A + 0.5 * C) * t) + B;
}
// -----------------------------------------------------------------------------
// Bicubic sample of colortex4 (history) – 4×4 taps → one RGB
// -----------------------------------------------------------------------------
vec3 OptimizedCubicTextureSample(vec2 P)
{
    // Pixel-space coordinate and intra-pixel fraction
    vec2 pixel = P * viewResolution + 0.5;
    vec2 frac = fract(pixel);
    ivec2 base = ivec2(pixel) - 1; // top-left of 4×4 stencil

    // --- horizontal cubic on four rows ---------------------------------------
    vec3 row0 = CubicHermite( // y = −1
        texelFetch(colortex4, base + ivec2(-1, -1), 0).rgb,
        texelFetch(colortex4, base + ivec2(0, -1), 0).rgb,
        texelFetch(colortex4, base + ivec2(1, -1), 0).rgb,
        texelFetch(colortex4, base + ivec2(2, -1), 0).rgb,
        frac.x);

    vec3 row1 = CubicHermite( // y =  0
        texelFetch(colortex4, base + ivec2(-1, 0), 0).rgb,
        texelFetch(colortex4, base + ivec2(0, 0), 0).rgb,
        texelFetch(colortex4, base + ivec2(1, 0), 0).rgb,
        texelFetch(colortex4, base + ivec2(2, 0), 0).rgb,
        frac.x);

    vec3 row2 = CubicHermite( // y = +1
        texelFetch(colortex4, base + ivec2(-1, 1), 0).rgb,
        texelFetch(colortex4, base + ivec2(0, 1), 0).rgb,
        texelFetch(colortex4, base + ivec2(1, 1), 0).rgb,
        texelFetch(colortex4, base + ivec2(2, 1), 0).rgb,
        frac.x);

    vec3 row3 = CubicHermite( // y = +2
        texelFetch(colortex4, base + ivec2(-1, 2), 0).rgb,
        texelFetch(colortex4, base + ivec2(0, 2), 0).rgb,
        texelFetch(colortex4, base + ivec2(1, 2), 0).rgb,
        texelFetch(colortex4, base + ivec2(2, 2), 0).rgb,
        frac.x);

    // --- vertical cubic through those rows -----------------------------------
    return CubicHermite(row0, row1, row2, row3, frac.y);
}

float MinDepth3x3(sampler2D tex, ivec2 fragCoord)
{
#if USE_GATHER
    vec2 invRes = 1.0 / viewResolution;
    vec2 tc = (vec2(fragCoord) + 0.5) * invRes;

    // (-1,-1) pack  → TL, TC, CL, C
    vec4 g0 = textureGatherOffset(tex, tc, ivec2(-1, -1), 0);

    // ( 0,-1) pack  → TC, TR, C,  CR
    vec4 g1 = textureGatherOffset(tex, tc, ivec2(0, -1), 0);

    // (-1, 0) pack  → CL, C,  BL, BC
    vec4 g2 = textureGatherOffset(tex, tc, ivec2(-1, 0), 0);

    // Min over the 12 returned components (9 unique texels + duplicates)
    float m = min(min(min(g0.x, g0.y), min(g0.z, g0.w)),
                  min(min(g1.x, g1.y), min(g1.z, g1.w)));
    m = min(m, min(min(g2.x, g2.y), min(g2.z, g2.w)));
    return m;

#else
    /* ---------- original 9-tap scalar fallback ---------- */
    float d0 = texelFetch(tex, fragCoord, 0).x;
    float d1 = texelFetch(tex, fragCoord + ivec2(-1, -1), 0).x;
    float d2 = texelFetch(tex, fragCoord + ivec2(0, -1), 0).x;
    float d3 = texelFetch(tex, fragCoord + ivec2(1, -1), 0).x;
    float d4 = texelFetch(tex, fragCoord + ivec2(-1, 0), 0).x;
    float d5 = texelFetch(tex, fragCoord + ivec2(1, 0), 0).x;
    float d6 = texelFetch(tex, fragCoord + ivec2(-1, 1), 0).x;
    float d7 = texelFetch(tex, fragCoord + ivec2(0, 1), 0).x;
    float d8 = texelFetch(tex, fragCoord + ivec2(1, 1), 0).x;

    float m = min(min(min(d0, d1), min(d2, d3)),
                  min(min(d4, d5), min(d6, min(d7, d8))));
    return m;
#endif
}

#ifdef DISTANT_HORIZONS
vec3 ReprojectDH2(vec3 position)
{
    bool isHand = position.z < 0.6;
    bool applyCameraMotion = (position.z < 1.0) && !isHand;

    position = ViewSpaceFromScreenSpace(position, dhProjectionInverse);
    position = toWorldSpaceNoLock(position);

    if (applyCameraMotion)
    {
        position += (cameraPosition - previousCameraPosition);
    }

    position = mat3(gbufferPreviousModelView) * position + gbufferPreviousModelView[3].xyz;
    position = ScreenSpaceFromViewSpace(position, dhPreviousProjection);

    return position;
}

#endif

/* RENDERTARGETS: 4 */
void main()
{
    vec3 color = vec3(0.0);
    vec2 uv = texcoord;
    ivec2 texel = ivec2(uv * viewResolution);

#ifdef TAA
    float z = texture(depthtex0, uv).x;
    #ifdef NO_ENTITY_TAA
    float entity = texture(colortex5, uv).g;
    #endif

    float offcenter_rej = TAA_OFFCENTER_REJECTION;
    float historyWeight = TAA_HISTORY_WEIGHT;

    float depth = MinDepth3x3(depthtex0, texel);
    vec3 current = texture2D(colortex0, uv + offsets[framemod8] * texelSize * 0.5).rgb;

    vec3 position = vec3(uv, depth);

    #ifndef WATER_PBR
    float alpha = texture(colortex1, uv).a;

    if (alpha > 0.1)
    {
        float depth2 = texelFetch(depthtex1, texel, 0).x;

        vec3 position2 = vec3(uv, depth2);

        position = mix(position2, position, pow(alpha, 10.0));
    }
    #endif

    vec3 reprojectedPosition = Reproject(position);

    #ifdef DISTANT_HORIZONS
    if (z >= 1.0)
    {
        depth = MinDepth3x3(dhDepthTex0, texel);
        position = vec3(uv, depth);
        reprojectedPosition = ReprojectDH2(position);
    }
    #endif

    #if TAA_FAST
    // Cheaper: 2×2 bilinear from previous mip (cuts 12 taps → 1)
    vec3 history = texture(colortex4, reprojectedPosition.xy).rgb;
    #else
    vec3 history = OptimizedCubicTextureSample(reprojectedPosition.xy);
    #endif
    bool isHand = position.z < 0.6;

    if (any(lessThan(reprojectedPosition.xy, vec2(0.0))) ||
        any(greaterThan(reprojectedPosition.xy, vec2(1.0))))
    {
        historyWeight = 0.0;
    }
    #ifdef NO_HAND_TAA
    if (isHand)
    {
        historyWeight = 0.0;
    }
    #endif
    #ifdef TAA_CLIP
    vec2 pcd = 1.0 - abs(2.0 * fract(reprojectedPosition.xy * viewResolution) - 1.0);
    historyWeight *= sqrt(pcd.x * pcd.y) * offcenter_rej + (1.0 - offcenter_rej);
    vec2 invRes = 1.0 / viewResolution;
    vec2 baseCoord = (vec2(texel) + 0.5) / viewResolution;

        /* ---------------------------------------------------------
         *  Neighbourhood min / max in YCoCg  (3×3 footprint)
         * --------------------------------------------------------- */
        #if !TAA_FAST
    // full 3×3 min/max loop (current version)
    vec3 minC = ENC_COL(current);
    vec3 maxC = minC;
    for (int y = -1; y <= 1; ++y)
        for (int x = -1; x <= 1; ++x)
        {
            vec3 nb = ENC_COL(texture(colortex0, baseCoord + vec2(x, y) * invRes).rgb);
            minC = min(minC, nb);
            maxC = max(maxC, nb);
        }
        #else
    // Fast: 4-tap box min/max (centre + 3 diagonal neighbours)
    vec3 minC = ENC_COL(current);
    vec3 maxC = minC;
    const ivec2 diag[3] = ivec2[3](ivec2(-1, -1), ivec2(1, -1), ivec2(-1, 1));
    for (int i = 0; i < 3; ++i)
    {
        vec3 nb = ENC_COL(texture(colortex0, baseCoord + vec2(diag[i]) * invRes).rgb);
        minC = min(minC, nb);
        maxC = max(maxC, nb);
    }
        #endif

    /* ---------------------------------------------------------
     *  Clamp history in the *same* colour space
     * --------------------------------------------------------- */
    vec3 currentYCoCg = ENC_COL(current);
    vec3 historyYCoCg = ENC_COL(history);

    historyYCoCg = clamp(historyYCoCg, minC, maxC);

        /* ---------------------------------------------------------
         *  Blend and convert back to RGB
         * --------------------------------------------------------- */

        #ifdef NO_ENTITY_TAA
    if (entity > 0.1)
    {
        historyWeight = clamp(1.0 - entity, 0.15, historyWeight);
    }
        #endif

    vec3 blendedYCoCg = mix(currentYCoCg, historyYCoCg, historyWeight);
    color = DEC_COL(blendedYCoCg);

    #else
    color = texture(colortex0, uv).rgb;
    #endif

    gl_FragData[0].rgb = clamp(color, 0.0001, 1000.0);
    gl_FragData[0].a = texture(colortex4, uv).a;
}
