#extension GL_ARB_gpu_shader5 : enable
#extension GL_ARB_shader_image_load_store : enable
#extension GL_ARB_shading_language_packing : enable
#extension GL_ARB_explicit_attrib_location : enable

#if defined BETTER_WATER_REFLECTIONS
    const bool colortex7MipmapEnabled = true;
    const bool colortex4MipmapEnabled = true;
#endif


#if defined BETTER_WATER_REFLECTIONS || defined SSR_QUALITY == 1    
    const bool colortex6MipmapEnabled = true;
#endif

flat varying vec3 ambientUp;
flat varying vec3 ambientDown;
flat varying vec3 ambientLeft;
flat varying vec3 ambientRight;
flat varying vec3 ambientB;
flat varying vec3 ambientF;
flat varying float ambientA;
varying vec2 texcoord;

layout(location = 0) out vec3 fragColor0;
#if SSR_QUALITY > 1
layout(location = 1) out vec4 fragColor1;
#endif

#if SSR_QUALITY > 1
/* RENDERTARGETS: 0,1 */
#else
/* RENDERTARGETS: 0 */
#endif

#include "/lib/light.glsl"

#if SHADOW_FILTERING == 2
    #include "/lib/shadows/shadowfunc_advanced.glsl"
#else
    #include "/lib/shadows/shadowfunc_simple.glsl"
#endif
#include "/lib/shadows/shadowfunc_no_shadow.glsl"

#if SSAO == 2
    #include "/lib/rtao/rtao_lib.glsl"
#elif SSAO == 1
    #include "/lib/rtao/rtao_lib_simple_new.glsl"
#endif

#include "/lib/reflection_lib_fake.glsl"

vec3 CalculateWaterCaustics(vec3 worldPos)
{
    float scale = worldPos.y / lightPos.y;
    vec2 pos = worldPos.xz - lightPos.xz * scale;
    vec2 coord0 = fract(pos / 64.0);
    float caustics = textureLod(colortex7, coord0, 0).g;
    return sunLight * caustics;
}

void main()
{
    // -------------------------------------------------------------------
    // 1. BATCH ALL TEXTURE FETCHES TOGETHER (reduce stalls)
    // -------------------------------------------------------------------
    ivec2 iTexCoords = ivec2(gl_FragCoord.xy);
    
    // Fetch all textures at once - GPU can pipeline these
    float z = texelFetch(depthtex0, iTexCoords, 0).x;
    vec3 albedo = texelFetch(colortex0, iTexCoords, 0).rgb;
    vec4 basicTex = texelFetch(colortex2, iTexCoords, 0);
    #ifdef PBR
        vec4 extendedTex = texelFetch(colortex3, iTexCoords, 0);
    #endif
    #if SSR_QUALITY > 1
        vec4 extra = texelFetch(colortex1, iTexCoords, 0);
    #endif
    
    // Early exit for sky - before doing any math
    if (z >= 1.0)
    {
        fragColor0 = texture(colortex8, texcoord).rgb;
        #ifndef IS_IRIS
            if (z >= 0.6) // not hand
                fragColor0 += texture(colortex2, texcoord).rgb;
        #endif
        return;
    }
    
    // -------------------------------------------------------------------
    // 2. DECODE ALL PROPERTIES (while textures are fetching)
    // -------------------------------------------------------------------
    bool hand = (z < 0.6);
    
    #if SSR_QUALITY > 1
        albedo += extra.rgb * extra.a;
    #endif
    
    vec3 normal = decodeNormal(basicTex.rg);
    vec2 decoded = decodeVec2(basicTex.a);
    vec2 lightmap = decodeVec2(basicTex.b);
    lightmap.x = lightmap.x * lightmap.x * lightmap.x;
    
    float ao = decoded.x;
    int temp = int(decoded.y * 255.0);
    float porosity = (temp <= 64) ? float(temp) / 64.0 : 0.0;
    float sss = (temp > 64) ? ((temp - 65) / 190.0) * 0.75 : 0.0;
    
    #ifdef LPV_ENABLED
        ao = (ao + 1.0) * 0.5;
    #endif
    
    // -------------------------------------------------------------------
    // 3. PBR PROPERTIES
    // -------------------------------------------------------------------
    vec3 reflectivity;
    float roughness;
    #ifdef PBR
        vec2 decoded2 = decodeVec2(extendedTex.x);
        float rainMult = sqrt(lightmap.y) * wetness * (1.0 - porosity * porosity);
        roughness = mix((1.0 - decoded2.y) * (1.0 - decoded2.y), 0.01, rainMult);
        reflectivity = vec3(decoded2.x);
    #else
        reflectivity = vec3(0.0);
        roughness = 1.0;
    #endif
    
    // -------------------------------------------------------------------
    // 4. POSITION CALCULATIONS
    // -------------------------------------------------------------------
    vec3 fragpos;
    #ifdef TAA
        fragpos = toScreenSpace(vec3(texcoord - (offsets[framemod8] * texelSize * 0.5), z));
    #else
        fragpos = toScreenSpace(vec3(texcoord, z));
    #endif
    
    vec3 p3 = toWorldSpace(fragpos);
    
    // -------------------------------------------------------------------
    // 5. LIGHTING SETUP
    // -------------------------------------------------------------------
    vec3 lightPosition = lightPos;
    #ifndef OVERWORLD
        lightPosition = vec3(0.0, 1.0, 0.0);
    #endif
    
    float diffuseSun = max(dot(normal, lightPosition), 0.0);
    vec3 colored = vec3(0.0);
    vec3 sAlbedo = vec3(0.6) * ambientA * lightmap.y;
    
    // -------------------------------------------------------------------
    // 6. MAIN LIGHTING CALCULATION
    // -------------------------------------------------------------------
    if (!hand)
    {
        // Batch texture lookups that might be needed
        float entity = texture(colortex5, texcoord).g;
        float noiseSample = noise_standard(gl_FragCoord.xy);
        
        #if SSAO > 0
            ao *= rtao(normal, fragpos);
        #else
            ao *= ao;
        #endif
        
        #ifdef OVERWORLD
            #ifdef SHADOWS_ENABLED
                diffuseSun = shadowFunc(diffuseSun, sss, p3, noiseSample, lightmap, fragpos, sAlbedo, albedo, normal, colored, entity);
            #else
                diffuseSun = shadowFunc_no_shadow(diffuseSun, sss, p3, lightmap, ao, fragpos);
            #endif
        #endif
        
        #ifdef CAUSTICS
            float prevmask = texture(colortex4, ReprojectWorld(p3).xy).a;
            if ((isEyeInWater == 0 && prevmask > 0.0) || (isEyeInWater == 1 && prevmask <= 0.0))
            {
                float strength = lightmap.y * max(diffuseSun, 0.1);
                colored += CalculateWaterCaustics(p3 + cameraPosition) * strength;
            }
        #endif
    }
    else
    {
        // Simplified hand lighting
        reflectivity = vec3(0.0);
        diffuseSun = max(dot(viewToWorld(normal), lightPosition), 0.0);
        diffuseSun *= min(eyeBrightnessSmooth.y * recip * lightmap.y, 1.0);
    }
    
    #ifdef HELD_LIGHT
        float heldLight = max(heldBlockLightValue, heldBlockLightValue2);
        float heldDistance = length(p3) / heldLight;
        lightmap.x = max(pow(clamp(1.0 - heldDistance, 0.0, 1.0), 1.5) * 0.75, lightmap.x);
    #endif
    
    // -------------------------------------------------------------------
    // 7. FINAL COMPOSITION
    // -------------------------------------------------------------------
    vec3 torchLighting = CalculateTorch(lightmap, normal, fragpos, ao);
    vec3 ambient = calcAmbientLight(normal, lightmap.xy, ambientUp, ambientDown, ambientRight, ambientLeft, ambientB, ambientF, sAlbedo + colored, diffuseSun, torchLighting) * ao;
    
    vec3 color = ambient * albedo;
    
    #if SSR_QUALITY <= 1
        #ifdef PBR
            if (reflectivity.y > 0.0)
            {
                color = max(computeReflections(lightmap.y, roughness, reflectivity, albedo, diffuseSun, normal, p3, color), vec3(0.0));
            }
        #endif
    #endif
    
    fragColor0 = color;
    #if SSR_QUALITY > 1
        fragColor1 = vec4(albedo, diffuseSun);
    #endif
}