varying vec2 texcoord;

#include "/lib/fog/fog3.glsl"

// -----------------------------------------------------------------------------
// HELPER FUNCTIONS
// -----------------------------------------------------------------------------

// Consolidated refraction data to minimize redundant calculations
struct RefractionData
{
    vec2 uvRefracted;
    vec2 uvTranslucent;
    float depth0;
    float depth1;
    ivec2 scenePixel;
    ivec2 transpPixel;
};

/** Cubic fall-off factor for screen-space border fog. */
float GetBorderFogMixFactor(in vec3 eyePlayerPos)
{
    float s = smoothstep(far * 0.5, far, length(eyePlayerPos));
    return s * s * s;
}

// Branchless bounds check - returns 1.0 if in bounds, 0.0 if out
float checkBounds(vec2 coord)
{
    vec2 bounded = step(vec2(0.0), coord) * step(coord, vec2(1.0));
    return bounded.x * bounded.y;
}

RefractionData calculateRefraction(vec2 uvOrig)
{
    vec2 uvRefr = uvOrig;
    vec2 uvTransp = uvOrig;

#ifdef REFRACTION
    // Fetch depth and normal textures once
    vec4 depthNormalData = texture(colortex13, uvOrig, 0);
    vec2 normalTex = depthNormalData.xy;
    float depth1 = ld(texture(depthtex1, uvOrig).x);

    vec2 rc = uvOrig + normalTex;

    // Branchless bounds check
    float inBounds = checkBounds(rc);

    // Water surface detection without branching
    float notInWater = 1.0 - float(isEyeInWater);

    // Calculate depth-based factor only for above water
    float z0_offset = ld(texture(depthtex0, rc).x);
    float z1_offset = ld(texture(depthtex1, rc).x);
    float depthDiff = abs(z0_offset - z1_offset) * 512.0;
    float depthFactor = clamp(depthDiff, 0.0, 1.0) * notInWater;

    // Combine factors: inBounds AND (underwater OR good depth difference)
    float refractionStrength = inBounds * max(1.0 - notInWater, depthFactor);
    rc = mix(uvOrig, rc, refractionStrength);

    // Border fog blend
    vec3 ssPos = toScreenSpace(vec3(rc, depth1));
    float borderFade = GetBorderFogMixFactor(ssPos);
    rc = mix(rc, uvOrig, borderFade);

    // Underwater distortion - branchless
    #ifdef UNDERWATER_DISTORTION
    float jitterAmount = texture(colortex7, rc).x * 0.005 * float(isEyeInWater);
    rc += jitterAmount;
    #endif

    // Final bounds check and assignment
    uvRefr = mix(uvOrig, rc, checkBounds(rc));

    // Calculate translucent UV with water falloff
    float wfalloff = mix(1.0, clamp(distance(uvRefr, uvOrig), 0.0, 1.0), notInWater);
    uvTransp = mix(uvOrig, uvRefr, wfalloff);
#endif

    // Pre-calculate pixel coordinates
    ivec2 scenePixel = ivec2(uvRefr * viewResolution);
    ivec2 transpPixel = ivec2(uvTransp * viewResolution);

    // Fetch final depth values at refracted coordinates
    float z0 = texelFetch(depthtex0, scenePixel, 0).x;
    float z1 = texelFetch(depthtex1, scenePixel, 0).x;

    return RefractionData(uvRefr, uvTransp, z0, z1, scenePixel, transpPixel);
}

// -----------------------------------------------------------------------------
// MAIN
// -----------------------------------------------------------------------------
void main()
{
    /* RENDERTARGETS: 0,4,13 */

    // ---------------------------------------------------------------------
    // 1. CONSOLIDATED COORDINATE & DEPTH SETUP
    // ---------------------------------------------------------------------
    vec2 uvOrig = texcoord;
    RefractionData refData = calculateRefraction(uvOrig);

    // ---------------------------------------------------------------------
    // 2. MASKS & FLAGS (Branchless)
    // ---------------------------------------------------------------------
    vec4 mask = texture(colortex5, texcoord);

    // Branchless water detection
    float waterThreshold = mix(0.5, 0.25, float(isEyeInWater));
    bool isWater = mask.r > waterThreshold;
    bool isGlassWater = (mask.b > 0.0 && mask.b < 0.01);
    bool isHand = (refData.depth0 < 0.6);

    // ---------------------------------------------------------------------
    // 3. DISTANT HORIZONS DEPTH MERGE
    // ---------------------------------------------------------------------
    float z0 = refData.depth0;
    float z1 = refData.depth1;

    // Group all depth-related fetches together for cache efficiency
#ifdef DISTANT_HORIZONS
    float z0DH = texture(dhDepthTex0, refData.uvRefracted).x;
    float z1DH = texture(dhDepthTex1, refData.uvRefracted).x;

    // Branchless depth selection
    float useZ0DH = step(1.0, z0);
    float useZ1DH = step(1.0, z1);
    z0 = mix(z0, z0DH, useZ0DH);
    z1 = mix(z1, z1DH, useZ1DH);

    vec3 fragPos0 = toScreenSpace_alt(refData.uvRefracted, refData.depth0, z0DH);
    vec3 fragPos1 = toScreenSpace_alt(refData.uvRefracted, refData.depth1, z1DH);
#else
    vec3 fragPos0 = toScreenSpace(vec3(refData.uvRefracted, z0));
    vec3 fragPos1 = toScreenSpace(vec3(refData.uvRefracted, z1));
#endif

    // ---------------------------------------------------------------------
    // 4. TEXTURE FETCHES (grouped for cache coherence)
    // ---------------------------------------------------------------------
    vec3 scene = texelFetch(colortex0, refData.scenePixel, 0).rgb;
    vec4 translucents = texelFetch(colortex1, refData.transpPixel, 0);
    vec4 sky = texture(colortex8, refData.uvTranslucent);

    // ---------------------------------------------------------------------
    // 5. CLOUD BLENDING (Branchless)
    // ---------------------------------------------------------------------
#ifdef CLOUDS
    #ifdef ALT_CLOUDS
    // Calculate cloud blend factor without branches
    float notFarPlane = step(z0, 0.999);
    float notHand = 1.0 - float(isHand);
    float hasCloudAlpha = 1.0 - sky.a;
    float cloudMask = notFarPlane * notHand * hasCloudAlpha;

    vec3 cloudContribution = sky.rgb * hasCloudAlpha;
    translucents.rgb = mix(translucents.rgb, cloudContribution, cloudMask);
    scene = mix(scene, cloudContribution, cloudMask);
    #endif
#endif

    // ---------------------------------------------------------------------
    // 6. VOLUMETRICS / TRANSLUCENTS BLEND
    // ---------------------------------------------------------------------
    // Always call VL function, let it handle early exit internally
    scene = getVolumetricRays3(scene, translucents, sky.rgb, fragPos0, fragPos1, isWater);

    // For non-overworld or specific conditions, add translucents
#ifndef OVERWORLD
    float addTranslucents = 1.0;
#else
    // In overworld: add translucents only when z0 >= 1.0 AND not in water
    float addTranslucents = step(1.0, z0) * (1.0 - float(isEyeInWater));
#endif
    scene += translucents.rgb * translucents.a * addTranslucents;

    // ---------------------------------------------------------------------
    // 7. DISTANCE FOG
    // ---------------------------------------------------------------------
    float fogMix = mix(darknessFactor, blindness, blindness);
    scene *= exp(-0.5 * length(fragPos0) * fogMix);

    // ---------------------------------------------------------------------
    // 8. OUTPUT
    // ---------------------------------------------------------------------
    gl_FragData[0].rgb = scene;

    // Consolidated water mask calculation
    float waterMasks = (isWater ? 0.5 : 0.0) + (isGlassWater ? 0.5 : 0.0);

    // Pass-through operations using original coordinates
#ifdef DISTANT_HORIZONS
    vec3 dhMask = vec3(0.0);
    #ifdef MC_GL_VENDOR_INTEL
    // Prevent refraction normals from showing up where they're not supposed to
    if (!(isWater || isGlassWater))
        dhMask.rg = vec2(0);
    #endif


    dhMask.b = encodeDepthBias(texture(dhDepthTex0, texcoord).x);
    gl_FragData[2].rgb = dhMask;
#endif

#ifdef TAA
    gl_FragData[1] = vec4(texture(colortex4, texcoord).rgb, waterMasks);
#else
    gl_FragData[1] = vec4(scene, waterMasks);
#endif
}