

#include "/lib/ssr/ssr_common.glsl"

vec3 RT_SIMPLE(vec3 dir, vec3 position, float roughness, float dither, float quality)
{
    vec3 rtPos = vec3(1.1);

    vec3 clipPosition = toClipSpace3(position);
    vec3 direction = normalize(toClipSpace3(position + dir) - clipPosition);
    vec3 maxLengths = (step(0.0, direction) - clipPosition) / direction;
    float mult = min(min(maxLengths.x, maxLengths.y), maxLengths.z);

    vec3 dirScaled = direction * mult;
    float invQuality = 1.0 / quality;

    // Determine the maximum lengths along each axis before exiting the clip space
    vec3 stepVector = direction * (mult / float(quality));

    float tolerance = abs(stepVector.z) + 0.01;

    for (int i = 0; i <= int(quality); ++i)
    {
        float stepFrac = (float(i) + dither) * invQuality;
        vec3 spos = clipPosition + dirScaled * stepFrac;

        ivec2 sampleCoord = ivec2(spos.xy / texelSize);
        float depthSample = texelFetch2D(depthtex0, sampleCoord, 0).r;

        if (depthSample < spos.z)
        {
            float dist = abs(depthSample - spos.z) / spos.z;
            if (dist <= tolerance)
            {
                rtPos = vec3(spos.xy, depthSample);
                break;
            }
        }
    }

    return rtPos;
}

const bool colortex0MipmapEnabled = true;
const bool colortex6MipmapEnabled = true;

#include "/lib/specular.glsl"

vec3 calculateSpecularLighting_simplified(
    vec3 normal,
    vec3 np3,
    vec3 albedo,
    float shading,
    float skylight,
    float roughness,
    vec3 f0,
    vec3 fragpos,
    vec3 scene, bool hand, inout vec3 a, inout vec3 b)
{
    roughness = max(0.0000001, roughness);

    // Compute metal index from f0.y
    int metalidx = int((f0.y * 255));
    bool isMetal = (metalidx > 229);
    f0 = CalculateF0(f0, isMetal, metalidx, albedo);

    float noise;
    if (!hand)
    {
        noise = blueNoise(gl_FragCoord.xy);
    }
    else
    {
        noise = 0.5;
    }
    if (!isMetal)
    {
        noise = mix(noise, 0.5, luma(f0));
    }
    //  roughness = 0.2;
    vec3 specTerm = shading * GGX2(normal, -np3, lightPos, roughness, f0) * pow(skylight, 3.0);

    // Indirect specular calculation
    vec3 indirectSpecular = vec3(0.0);
    vec3 fresnelDiffuse = vec3(0.0);

    // Coordinate basis for sampling
    mat3 basis = CoordBase(normal);
    vec3 normSpaceView = normalize(-np3 * basis);

    vec3 reflectionColorScene = scene;

    // Sample  VNDF
    // vec3 H = SampleBeckmannVNDF(normSpaceView, roughness, blueNoise4(gl_FragCoord.xy).xy);
    vec3 H = SampleGGXVNDF2(normSpaceView, roughness, noise);
    vec3 Ln = reflect(-normSpaceView, H);
    vec3 L = basis * Ln;
    float cosThetaI = dot(np3, normal);
    float clampedCosThetaI = clamp(1.0 + cosThetaI, 0.0, 1.0);
    float F2 = pow(clampedCosThetaI, 8.0);

    // Fresnel term
    vec3 F = f0 + (1.0 - f0) * pow(clamp(1.0 + dot(-Ln, H), 0.0, 1.0), 5.0);

    // Geometry term
    float NoL = clamp(dot(normal, L), 0.0, 1.0);
    float G1 = g(NoL, roughness);

    // Ray contribution
    vec3 rayContrib = F * G1;
    float lod = mix(1.0, 8.0, clamp(roughness + F2, 0, 1));

    float rayQuality = (clamp(32 * sqrt(luma(rayContrib)), 1.0, REFLECTION_SAMPLES_MAX));
    vec4 sky = skyFromTexLod(L, colortex6, lod);
    sky.rgb = mix(sky.rgb, vec3(sky.a * sunLight), clamp(sky.a, 0, 1));

    // float bounceAmount = step(0.0, L.y) + clamp(-L.y * 0.1 + 0.1, 0.0, 1.0);
    float bounceAmount = clamp(L.y * 0.5 + 0.5, 0.0, 1.0) * sqrt(skylight);

    bounceAmount *= (!isMetal && roughness > 0.5) ? 0.5 : 1.0;
    vec3 reflectionColor = mix(reflectionColorScene, sky.rgb * bounceAmount, pow(skylight, 3.0));

    if (!hand && luma(rayContrib) > 0.02 && (rayQuality > 5.0) && (roughness < 0.3))
    {

        vec3 rtPos = RT_SIMPLE(mat3(gbufferModelView) * L, fragpos, roughness, noise, rayQuality);

        if (rtPos.z < 1.0)
        {
            reflectionColor = textureLod(colortex4, rtPos.xy, lod).rgb;
        }
    }

    indirectSpecular += reflectionColor * rayContrib;
    fresnelDiffuse += rayContrib;

    a = (indirectSpecular) + (specTerm * sunLight * SPECULAR_BOOST);
    b = (isMetal ? vec3(0.0) : (vec3(1.0) - fresnelDiffuse));

    vec3 effectiveAlbedo = (metalidx < 229) ? vec3(1.0) : albedo;
    vec3 finalout = vec3((a + b * scene) * effectiveAlbedo);

    return finalout;
}

vec3 computeReflections(vec3 albedo, float lightmapY, float smoothness, vec3 f0, vec3 normal, vec3 fragpos, float materialsX, vec3 scene, bool hand, inout vec3 a, inout vec3 b)
{

    vec3 p3 = toWorldSpace(fragpos);

    float roughness = unpackRoughness(smoothness);
    float shading = texture(colortex1, texcoord).a;
    vec3 reflections = calculateSpecularLighting_simplified(normal, normalize(p3), albedo, shading, lightmapY, roughness, f0, fragpos, scene, hand, a, b);

    return vec3(reflections);
}
