void decodeMaterial(
    sampler2D inputTex1,   // Texture for basic material properties
    ivec2 texCoords,       // Texture coordinates for inputTex1
    sampler2D inputTex2,   // Texture for extended material properties (used if PBR is enabled)
    vec2 texCoord,         // Texture coordinates for inputTex2
    float wetness,         // Wetness factor affecting roughness
    vec3 albedo,           // Base color of the material
    out vec3 normal,       // Output normal vector
    out float ao,          // Output ambient occlusion
    out float sss,         // Output subsurface scattering factor
    out vec2 lightmap,     // Output lightmap coordinates
    out vec3 reflectivity, // Output reflectivity (only if PBR is enabled)
    out float roughness    // Output roughness (only if PBR is enabled)
)
{
    // Fetch basic material properties from inputTex1.
    vec4 basicTex = texelFetch(inputTex1, texCoords, 0);

    // Decode normal vector using the red and green channels.
    normal = decodeNormal(basicTex.rg);

    // Decode lightmap coordinates from the blue channel.
    lightmap = decodeVec2(basicTex.b);
    lightmap.x = lightmap.x * lightmap.x * lightmap.x;

    // Decode additional properties from the alpha channel.
    vec2 decoded = decodeVec2(basicTex.a);

    // Calculate subsurface scattering (sss) and porosity.
    sss = max((decoded.y - 0.252941) / 0.747059, 0.0) * 0.75;
    float porosity = min(decoded.y / 0.252941, 1.0);

    // Ambient occlusion is stored in decoded.x.
    ao = decoded.x;

#ifdef LPV_ENABLED
    // Replace mix(ao, 1.0, 0.5) with a simple arithmetic average.
    ao = (ao + 1.0) * 0.5;
#endif

#ifdef PBR
    // Fetch extended material properties from inputTex2.
    vec4 extendedTex = texture(inputTex2, texCoord);
    vec2 decoded2 = decodeVec2(extendedTex.x);

    // Calculate rain multiplier using sqrt() and squaring of porosity.
    float rainMult = sqrt(lightmap.y) * wetness * (1.0 - porosity * porosity);

    // Replace pow(1.0 - decoded2.y, 2.0) with a multiplication.
    float baseRoughness = (1.0 - decoded2.y) * (1.0 - decoded2.y);
    roughness = mix(baseRoughness, 0.01, rainMult);

    // Set reflectivity based on the decoded value.
    reflectivity = vec3(decoded2.x);
#else
    // If PBR is not enabled, assign default values.
    reflectivity = vec3(0.0);
    roughness = 1.0;
#endif
}
